import os
import yaml
import numpy as np
import argparse


def build_lattice(size, boundary='periodic'):
    """
    Build a 2D lattice of given size and boundary conditions.
    Returns a list of links: [((x, y), mu), ...]
    where mu=0 for x-direction, mu=1 for y-direction.
    """
    links = []
    # Define directional offsets for mu=0 (x), mu=1 (y)
    directions = [(1, 0), (0, 1)]

    for x in range(size):
        for y in range(size):
            for mu, (dx, dy) in enumerate(directions):
                nx = x + dx
                ny = y + dy
                if boundary == 'periodic':
                    nx %= size
                    ny %= size
                else:
                    if not (0 <= nx < size and 0 <= ny < size):
                        continue
                links.append(((x, y), mu))
    return links


def main(config_path):
    # Load configuration
    with open(config_path) as f:
        cfg = yaml.safe_load(f)

    size = cfg.get('lattice_size', 4)
    bc = cfg.get('boundary_conditions', 'periodic')
    # Resolve data_dir relative to the directory of the config file
    base_dir = os.path.dirname(os.path.abspath(config_path))
    data_dir_cfg = cfg.get('data_dir', 'data')
    if os.path.isabs(data_dir_cfg):
        data_dir = data_dir_cfg
    else:
        data_dir = os.path.join(base_dir, data_dir_cfg)

    # Build lattice
    lattice = build_lattice(size, bc)

    # Ensure data directory exists
    os.makedirs(data_dir, exist_ok=True)

    # Save lattice links to a NumPy file
    lattice_path = os.path.join(data_dir, 'lattice.npy')
    np.save(lattice_path, np.array(lattice, dtype=object))
    print(f'Lattice with {len(lattice)} links saved to {lattice_path}')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Build lattice links')
    parser.add_argument('--config', default='config.yaml', help='Path to config file')
    args = parser.parse_args()
    main(args.config)